<?php

namespace App\Models;

use CodeIgniter\Model;
use DateTime;
use DateInterval;

class RecuperacionModel extends Model
{
    protected $table            = 'recuperacion';
    protected $primaryKey       = 'id';
    protected $useAutoIncrement = true;

    protected $returnType       = 'array';
    protected $useSoftDeletes   = true;

    protected $allowedFields = [
        'uuid',
        'usuario_id',
        'ip',
        'fecha_uso',
        'fecha_vigencia',
    ];

    protected $useTimestamps = true;
    protected $createdField  = 'created_at';
    protected $updatedField  = 'modified_at';
    protected $deletedField  = 'deleted_at';

    protected $beforeInsert = ['generateUUID'];

    /**
     * Hook para generar el UUID.
     */
    protected function generateUUID(array $data)
    {
        if (!isset($data['data']['uuid']) || empty($data['data']['uuid'])) {
            $data['data']['uuid'] = $this->createUUID();
        }

        return $data;
    }

    /**
     * Genera un UUID v4 (sin dependencias).
     */
    private function createUUID()
    {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0xffff)
        );
    }

    /**
     * ✔ Crear token de recuperación y asignar vigencia de 24 horas
     */
    public function crearToken(int $usuarioId, string $ip = null)
    {
        // fecha actual
        $now = new DateTime();

        // sumar 24 horas
        $vigencia = (clone $now)->add(new DateInterval('PT24H'));

        $data = [
            'usuario_id'     => $usuarioId,
            'ip'             => $ip,
            'fecha_uso'      => null,
            'fecha_vigencia' => $vigencia->format('Y-m-d H:i:s'),
        ];

        $this->insert($data);

        // Obtener ID recién insertado
        $id = $this->getInsertID();

        // Consultar registro para obtener UUID generado en beforeInsert
        $row = $this->find($id);

        return $row['uuid'];

    }

    /**
     * ✔ Verifica si un token es válido
     * Reglas:
     * 1. fecha_uso es NULL
     * 2. fecha_vigencia >= ahora (no han pasado 24 hrs)
     */
    public function esTokenValido(string $uuid)
    {
        $token = $this->where('uuid', $uuid)->first();

        if (!$token) {
            return false;
        }

        // 1. Token NO usado
        if (!is_null($token['fecha_uso'])) {
            return false;
        }

        // 2. Vigencia
        $vigencia = new DateTime($token['fecha_vigencia']);
        $now      = new DateTime();

        return $vigencia >= $now;
    }
}
